<?php
/**
 * Handles the modifications and registration of PRO models.
 *
 * @since 4.7.8
 *
 * @package Tribe\Events\Pro\Models
 */


namespace Tribe\Events\Pro\Models;

use WP_Post;
use Tribe__Events__Pro__Geo_Loc as Geolocalization;

/**
 * Class Service_Provider
 *
 * @since 4.7.8
 *
 * @package Tribe\Events\Pro\Models
 */
class Service_Provider extends \tad_DI52_ServiceProvider {

	/**
	 * Binds and sets up implementations.
	 *
	 * @since 4.7.8
	 */
	public function register() {
		// Register this provider in the service locator to allow unhooking actions hooked by it.
		$this->container->singleton( 'events-pro.models', $this );

		add_filter( 'tribe_get_event', [ $this, 'filter_tribe_get_event' ] );
		add_filter( 'tribe_get_venue_object', [ $this, 'filter_tribe_get_venue_object' ] );
	}

	/**
	 * Filters the decorated event post object to add some PRO flags.
	 *
	 * @since 4.7.8
	 *
	 * @param \WP_Post $post The decorated event post, as generated by the `tribe_get_event` function.
	 *
	 * @return \WP_Post The decorated event post, as generated by the `tribe_get_event` function with additional PRO
	 *                  properties.
	 */
	public function filter_tribe_get_event( \WP_Post $post ) {
		$post->recurring = tribe_is_recurring_event( $post->ID );

		return $post;
	}

	/**
	 * Filters the decorated venue post object to add some PRO flags.
	 *
	 * @since 4.7.8
	 *
	 * @param \WP_Post $post The decorated event post, as generated by the `tribe_get_venue_object` function.
	 *
	 * @return \WP_Post The decorated event post, as generated by the `tribe_get_venue_object` function with additional PRO
	 *                  properties.
	 */
	public function filter_tribe_get_venue_object( WP_Post $post ) {
		$post->geolocation = (object) [
			'overwrite_coordinates' => tribe_is_truthy( get_post_meta( $post->ID, Geolocalization::OVERWRITE, true ) ),
			'latitude'              => get_post_meta( $post->ID, Geolocalization::LAT, true ),
			'longitude'             => get_post_meta( $post->ID, Geolocalization::LNG, true ),
			'address'               => get_post_meta( $post->ID, Geolocalization::ADDRESS, true ),
			/**
			 * @todo  @lucatume add value here when doing a location search.
			 */
			'distance'              => false,
		];

		return $post;
	}
}
